<?php
// Set timezone to US/Pacific
date_default_timezone_set('America/Los_Angeles');
// plot2day.php: Generate PNG plot of temperature(s) (F) and humidity (%) vs time from CSV
// Handles ISO datetime in first column, variable temp columns, and incomplete data

$csv = 'humidistat_data.csv';
if (!file_exists($csv)) {
    header('Content-Type: text/plain');
    die('CSV file not found');
}

$now = time();
$period = $_GET['period'] ?? '2day';
switch ($period) {
    case '1hour':
        $start_ts = $now - 3600;
        $period_label = 'Last Hour';
        break;
    case '4hour':
        $start_ts = $now - 4 * 3600;
        $period_label = 'Last 4 Hours';
        break;
    case '2day':
    default:
        $start_ts = $now - 2 * 86400;
        $period_label = 'Last 2 Days';
        break;
}

// Data arrays
$dates = [];
$tempsF = [];
$humids = [];

$fh = fopen($csv, 'r');
$malformed = fopen('/tmp/malformed.txt', 'a');
$prev_ts = null;
while (($row = fgetcsv($fh)) !== false) {
    if (count($row) < 4) {
        fwrite($malformed, implode(',', $row) . "\n");
        continue;
    }
    // CSV format: timestamp,tempC_bmp,tempC_aht,humidity,...
    $dateStr = $row[0]; // full timestamp in PST
    $tempC_bmp = floatval($row[1]);
    $humid = floatval($row[3]);
    if (empty($dateStr) || strpos($dateStr, "\0") !== false) {
        fwrite($malformed, implode(',', $row) . "\n");
        continue;
    }
    // Parse as PST
    $dt_pst = DateTime::createFromFormat('Y-m-d H:i:s', $dateStr, new DateTimeZone('America/Los_Angeles'));
    if (!$dt_pst) {
        fwrite($malformed, implode(',', $row) . "\n");
        continue;
    }
    $ts = $dt_pst->getTimestamp();
    // Only flag non-monotonic if previous valid row exists
    if ($prev_ts !== null && $ts < $prev_ts) {
        fwrite($malformed, "NON-MONOTONIC:" . implode(',', $row) . "\n");
        continue;
    }
    $prev_ts = $ts;
    // Only include data within the selected period
    if ($ts < $start_ts) {
        continue;
    }
    $dates[] = $dt_pst->format('Y-m-d H:i:s'); // now in Pacific time
    // Only plot tempC_bmp
    if ($tempC_bmp == 0 || $tempC_bmp > 45) {
        $tempsF[] = null;
    } else {
        $tempsF[] = $tempC_bmp * 9/5 + 32;
    }
    $humids[] = $humid;
}
fclose($fh);
fclose($malformed);

$n = count($dates);
if ($n < 1) {
    header('Content-Type: text/plain');
    die("No data for $period_label");
}

// Compute Y axis min/max for temperature and humidity separately
$allTemps = array_filter($tempsF, function($tf) { return $tf !== null; });
$allHumids = array_filter($humids, function($h) { return $h !== null; });
$minT = $maxT = null;
if (count($allTemps) > 0) {
    $minT = min($allTemps);
    $maxT = max($allTemps);
}
if ($minT === null || $maxT === null) {
    $minT = 0;
    $maxT = 100;
}
$rangeT = $maxT - $minT;
if ($rangeT == 0) {
    $rangeT = 1;
    $minT -= 0.5;
    $maxT += 0.5;
}
$minT = floatval($minT);
$maxT = floatval($maxT);

$minH = $maxH = null;
if (count($allHumids) > 0) {
    $minH = min($allHumids);
    $maxH = max($allHumids);
}
if ($minH === null || $maxH === null) {
    $minH = 0;
    $maxH = 100;
}
$rangeH = $maxH - $minH;
if ($rangeH == 0) {
    $rangeH = 1;
    $minH -= 0.5;
    $maxH += 0.5;
}
$minH = floatval($minH);
$maxH = floatval($maxH);

$w = 800;
$h = 400;
$margin = 50;
$im = imagecreatetruecolor($w, $h);
$white = imagecolorallocate($im, 255,255,255);
$black = imagecolorallocate($im, 0,0,0);
$blue = imagecolorallocate($im, 0,0,255);
$green = imagecolorallocate($im, 0,180,0);
$gray = imagecolorallocate($im, 220,220,220);
imagefill($im, 0, 0, $white);

// Title
imagestring($im, 4, $w/2 - 100, 10, "Temperature & Humidity - $period_label", $black);

// Y axis: temperature (left), humidity (right)
$tickStepT = ($maxT - $minT) > 40 ? 10 : 5;
$tickStepH = ($maxH - $minH) > 40 ? 10 : 5;
// Temperature Y axis (left)
for ($yf = ceil($minT/$tickStepT)*$tickStepT; $yf <= $maxT; $yf += $tickStepT) {
    $y = $h-$margin - ($yf-$minT)*($h-2*$margin)/($maxT-$minT);
    imageline($im, $margin-5, $y, $margin, $y, $black);
    imagestring($im, 2, 8, $y-7, sprintf('%.1f', $yf), $black);
    imageline($im, $margin, $y, $w-$margin, $y, $gray);
}
// Humidity Y axis (right)
for ($yh = ceil($minH/$tickStepH)*$tickStepH; $yh <= $maxH; $yh += $tickStepH) {
    $y = $h-$margin - ($yh-$minH)*($h-2*$margin)/($maxH-$minH);
    imageline($im, $w-$margin, $y, $w-$margin+5, $y, $green);
    imagestring($im, 2, $w-$margin+10, $y-7, sprintf('%.1f', $yh), $green);
}
// Draw axes
imageline($im, $margin, $margin, $margin, $h-$margin, $black); // Y temp
imageline($im, $w-$margin, $margin, $w-$margin, $h-$margin, $green); // Y humidity
imageline($im, $margin, $h-$margin, $w-$margin, $h-$margin, $black); // X

// X axis: always 2 days, ticks every 4 hours
$plotW = $w - 2*$margin;
$plotH = $h - 2*$margin;
$startTs = min(array_map('strtotime', $dates));
$endTs = max(array_map('strtotime', $dates));
if ($startTs == $endTs) $endTs = $startTs + 86400; // avoid zero range
$interval = 4*60*60;
$firstTick = ceil($startTs/$interval)*$interval;
for ($tick = $firstTick; $tick <= $endTs; $tick += $interval) {
    $frac = ($tick - $startTs) / ($endTs - $startTs);
    $x = $margin + $frac * $plotW;
    imageline($im, $x, $h-$margin, $x, $h-$margin+5, $black);
    imageline($im, $x, $margin, $x, $h-$margin, $black);
    $label = date('H:i', $tick);
    imagestring($im, 2, $x-14, $h-$margin+8, $label, $black);
}
$midnightTick = strtotime(date('Y-m-d 00:00:00', $startTs));
while ($midnightTick <= $endTs) {
    $frac = ($midnightTick - $startTs) / ($endTs - $startTs);
    $x = round($margin + $frac * $plotW);
    if ($x > $margin + 2) {
        $barColor = imagecolorallocate($im, 40, 40, 40);
        imageline($im, $x-1, $margin, $x-1, $h-$margin, $barColor);
        imageline($im, $x,   $margin, $x,   $h-$margin, $barColor);
        imageline($im, $x+1, $margin, $x+1, $h-$margin, $barColor);
    }
    $midnightTick = strtotime('+1 day', $midnightTick);
}

// Plot tempC_bmp (left axis, blue)
imagestring($im, 3, $margin+10, $margin+10, 'Temperature (°F)', $blue);
for ($i=1; $i<$n; $i++) {
    $ts1 = strtotime($dates[$i-1]);
    $ts2 = strtotime($dates[$i]);
    $x1 = $margin + ($ts1-$startTs)*($w-2*$margin)/($endTs-$startTs);
    $x2 = $margin + ($ts2-$startTs)*($w-2*$margin)/($endTs-$startTs);
    $y1 = $tempsF[$i-1] !== null ? $h-$margin - ($tempsF[$i-1]-$minT)*($h-2*$margin)/($maxT-$minT) : null;
    $y2 = $tempsF[$i] !== null ? $h-$margin - ($tempsF[$i]-$minT)*($h-2*$margin)/($maxT-$minT) : null;
    if ($y1 !== null && $y2 !== null) {
        imageline($im, $x1, $y1, $x2, $y2, $blue);
    }
}
// Plot humidity (right axis, green)
imagestring($im, 3, $w-$margin-120, $margin+10, 'Humidity (%)', $green);
for ($i=1; $i<$n; $i++) {
    $ts1 = strtotime($dates[$i-1]);
    $ts2 = strtotime($dates[$i]);
    $x1 = $margin + ($ts1-$startTs)*($w-2*$margin)/($endTs-$startTs);
    $x2 = $margin + ($ts2-$startTs)*($w-2*$margin)/($endTs-$startTs);
    $h1 = $h-$margin - ($humids[$i-1]-$minH)*($h-2*$margin)/($maxH-$minH);
    $h2 = $h-$margin - ($humids[$i]-$minH)*($h-2*$margin)/($maxH-$minH);
    imageline($im, $x1, $h1, $x2, $h2, $green);
}
// X labels (start/end date)
imagestring($im, 2, $margin, $h-20, date('Y-m-d H:i:s', $startTs), $black);
imagestring($im, 2, $w-$margin-100, $h-20, date('Y-m-d H:i:s', $endTs), $black);
imagestring($im, 3, $w/2 - 20, $h - 35, 'Time', $black);
// Output PNG
header('Content-Type: image/png');
imagepng($im);
imagedestroy($im);
?>
