<?php
header("Cache-Control: no-cache, no-store, must-revalidate");
header("Pragma: no-cache");
header("Expires: 0");
error_reporting(E_ALL);
ini_set('display_errors', 1);

$settingsFile = "settings.json";
$readingsFile = "current.readings";

// Default user settings (configurable)
$defaultSettings = [
    "thermostat" => "113",  // °F
    "humidity" => "50"
];

// Default sensor readings (read-only)
$defaultReadings = [
    "indoor_temp" => "N/A",
    "indoor_humid" => "N/A",
    "Twater" => "N/A"
];

// Load settings from JSON file
$settings = file_exists($settingsFile) ? json_decode(file_get_contents($settingsFile), true) : $defaultSettings;

// Load sensor readings from JSON file
$readings = file_exists($readingsFile) ? json_decode(file_get_contents($readingsFile), true) : $defaultReadings;

// Handle AJAX POST requests (update settings only)
if ($_SERVER["REQUEST_METHOD"] === "POST") {
    $response = "❌ Error: Unable to save setting.";

    if (isset($_POST["setting"]) && isset($_POST["value"])) {
        $settingType = $_POST["setting"];
        $settingValue = $_POST["value"];

        // Only allow changes for user settings, not sensor data
        if (array_key_exists($settingType, $defaultSettings)) {
            $settings[$settingType] = $settingValue;
            if (file_put_contents($settingsFile, json_encode($settings, JSON_PRETTY_PRINT)) !== false) {
                $response = "✅ Saved: " . htmlspecialchars($settingType) . " = " . htmlspecialchars($settingValue);
            }
        }
    }

    echo $response;
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Salubria3</title>
    <meta http-equiv="Cache-Control" content="no-cache, no-store, must-revalidate">
    <meta http-equiv="Pragma" content="no-cache">
    <meta http-equiv="Expires" content="0">
    <style>
        body { font-family: sans-serif; text-align: center; }
        .container { margin: 20px auto; width: 50%; padding: 20px; background: #f0f0f0; border-radius: 10px; }
        .button-group { display: flex; justify-content: center; gap: 10px; }
        button { padding: 10px 20px; font-size: 16px; cursor: pointer; border: none; border-radius: 5px; background: grey; color: white; }
        .button-group button.active { background: blue; }
        .control-mode button.active { background: blue; }
        .heater-control button.active { background: red; }
    </style>
</head>
<body>

    <div class="container">
    <h1>Salubria</h1>




    <div class="thermostat-control" style="display: flex; align-items: center; gap: 10px; margin: 10px 0;">
        <span>Thermostat:</span>
        <input type="number" id="thermostatInput" value="<?php echo htmlspecialchars($settings["thermostat"]); ?>" style="width: 80px;">
        <span id="currentTemp" style="font-weight: bold;">(...°F)</span>
        <button id="thermostatSave" onclick="saveSetting('thermostat', 'thermostatInput')">Save</button>
        <span id="furnaceStatus" style="font-weight: bold;">Furnace: ...</span>
    </div>

    <div class="humidity-control" style="display: flex; align-items: center; gap: 10px; margin: 10px 0;">
        <span>Humidity:</span>
        <input type="number" id="humidityInput" value="<?php echo htmlspecialchars($settings["humidity"]); ?>" style="width: 80px;">
        <span id="currentHumid" style="font-weight: bold;">(...%)</span>
        <button id="humiditySave" onclick="saveSetting('humidity', 'humidityInput')">Save</button>
        <span id="dehumidifierStatus" style="font-weight: bold;">Dehumidifier: ...</span>
    </div>

    <div class="control-mode">
        <div class="valve-label">Control Mode:</div>
        <button onclick="setControlMode('web_thermostat')">Web Control</button>
        <button onclick="setControlMode('remote_thermostat')">Remote Control</button>
    </div>

</div>

<!-- Plot of last 2 days of T and H -->
<div style="text-align:center; margin-top:30px;">
    <select id="plotPeriod" onchange="updatePlot()">
        <option value="1hour">Last Hour</option>
        <option value="4hour">Last 4 Hours</option>
        <option value="2day" selected>Last 2 Days</option>
    </select>
    <br>
    <img id="plotImg" src="plot2day.php?t=<?=time()?>" alt="Temperature & Humidity Plot" style="max-width:100%; border:1px solid #888; box-shadow: 2px 2px 8px #ccc;">
</div>

<script>
    function saveSetting(setting, inputId) {
        let value = parseFloat(document.getElementById(inputId).value);
        let button = document.getElementById(setting + 'Save');
        let originalText = button.textContent;
        button.textContent = "Saving...";
        let url = "http://192.168.1.184/setpoint";
        let body = "";
        if (setting === "thermostat") {
            value = (value - 32) * 5 / 9;  // Convert °F to °C
            body = `valueC=${encodeURIComponent(value.toFixed(2))}`;
        } else if (setting === "humidity") {
            body = `humidity=${encodeURIComponent(value)}`;
        }
        fetch(url, {
            method: "POST",
            headers: { "Content-Type": "application/x-www-form-urlencoded" },
            body: body
        })
        .then(response => response.json())
        .then(data => {
            console.log("ESP32 response:", data);
            button.textContent = "Saved!";
            setTimeout(() => button.textContent = originalText, 2000);
        })
        .catch(err => {
            console.error("ESP32 error:", err);
            button.textContent = "Error";
            setTimeout(() => button.textContent = originalText, 2000);
        });
    }

    function setControlMode(mode) {
        fetch("http://192.168.1.184/control", {
            method: "POST",
            headers: { "Content-Type": "application/x-www-form-urlencoded" },
            body: `state=${mode}`
        })
        .then(response => response.text())
        .then(data => {
            console.log("Control mode set:", data);
            if (mode === 'web_thermostat') {
                // Auto-save current settings to PHP when switching to web mode
                let thermoValue = document.getElementById('thermostatInput').value;
                let humidValue = document.getElementById('humidityInput').value;
                fetch(window.location.href, {
                    method: "POST",
                    headers: { "Content-Type": "application/x-www-form-urlencoded" },
                    body: `setting=thermostat&value=${encodeURIComponent(thermoValue)}`
                }).then(() => console.log("Thermostat setting saved to PHP"));
                fetch(window.location.href, {
                    method: "POST",
                    headers: { "Content-Type": "application/x-www-form-urlencoded" },
                    body: `setting=humidity&value=${encodeURIComponent(humidValue)}`
                }).then(() => console.log("Humidity setting saved to PHP"));
            }
        })
        .catch(err => console.error("Control mode error:", err));
    }

    // Fetch current temp/humidity from ESP32
    function fetchStatus() {
        fetch("http://192.168.1.184/status")
            .then(response => response.json())
            .then(data => {
                if (data.tempC_aht !== undefined) {
                    let tempF = (data.tempC_aht * 9 / 5) + 32;
                    document.getElementById("currentTemp").textContent = tempF.toFixed(1) + "°F";
                }
                if (data.humidity !== undefined) {
                    document.getElementById("currentHumid").textContent = data.humidity.toFixed(1) + "%";
                }
                // Highlight active control mode
                let webBtn = document.querySelector('button[onclick*="web_thermostat"]');
                let remoteBtn = document.querySelector('button[onclick*="remote_thermostat"]');
                if (data.control === 'web_thermostat') {
                    webBtn.classList.add('active');
                    remoteBtn.classList.remove('active');
                } else if (data.control === 'remote_thermostat') {
                    remoteBtn.classList.add('active');
                    webBtn.classList.remove('active');
                }
                // Update furnace and dehumidifier status
                document.getElementById("furnaceStatus").textContent = "Furnace: " + data.furnace;
                document.getElementById("dehumidifierStatus").textContent = "Dehumidifier: " + data.dehumidifier;
            })
            .catch(err => {
                document.getElementById("currentTemp").textContent = "(?°F)";
                document.getElementById("currentHumid").textContent = "?%";
            });
    }
    setInterval(fetchStatus, 5000); // Update every 5 seconds
    window.onload = function() {
        fetchStatus();
    };

    function updatePlot() {
        let period = document.getElementById('plotPeriod').value;
        let img = document.getElementById('plotImg');
        img.src = `plot2day.php?period=${period}&t=${Date.now()}`;
    }
</script>

</body>
</html>
